function MapManager() {

	var self = this;
	
	// DOM properties
	this.container  = document.getElementById('livemap-container');
	this.buildLayer = document.getElementById('canvas-layer-structures');
	this.pavedLayer = document.getElementById('canvas-layer-roads');
	this.gridLayer  = document.getElementById('canvas-layer-grid');
	this.standLayer = document.getElementById('canvas-layer-standings');
	this.planLayer  = document.getElementById('canvas-layer-planner');
	this.mouseLayer = document.getElementById('canvas-layer-mouse');
	
	// Functional properties
	this.showPlayers = false;
	this.showGuildClaims = true;
	this.showPersonalClaims = false;
	this.showAdminLands = false;
	this.defaultImage = true;
	this.zoomScale = 1;
	this.curYPos = 0;
	this.curXPos = 0;
	this.curDown = false;
	this.wheelZoom = true;
	this.plannerBool = false;
	this.mouseX = 0;
	this.mouseY = 0;
	this.mouseR = 75;
	
	// Data properties
	this.structures = [];
	this.pavedTiles = {};
	this.flatTiles  = [];
	this.regions    = [];
	
	// Toggle loading data animation
	this.showLoader = function() { document.getElementById('loader').style.display = 'block'; };
	this.hideLoader = function() { document.getElementById('loader').style.display = 'none'; };
	
	// Get guild info by ID
	this.getGuildInfoById = function(id) {
		for( var i = 0; i < claims.length; i++ ) if( claims[i].id === id ) return claims[i];
		console.log("Error: Couldn't find guild by ID: " + id);
		return false;
	}
	
	// Toggle player markers
	this.togglePlayers = function() {
		
		$('#ctrlbtn-players').toggleClass('control-button-inactive');
		
		this.showPlayers = this.showPlayers ? false : true;
		if( this.showPlayers ) this.drawPlayerMarkers();
		else this.removePlayerMarkers();
		
	};
	
	this.drawPlayerMarkers = function() {
		
		var players = Master.onlinePlayers;
	
		for( var i = 0; i < players.length; i++ ) {
			// Name
			var name = players[i].FirstName + ' ' + players[i].LastName;
			// Draw marker
			var marker = new Image();
			marker.src = 'img/player-marker.png';
			marker.className = 'marker-player';
			marker.id = 'marker-player-' + i;
			marker.style.setProperty('left', (players[i].x * this.zoomScale - 8) + 'px' );
			marker.style.setProperty('top', (players[i].y * this.zoomScale - 16) + 'px' );
			this.container.appendChild(marker);
			// Draw name label
			var label = document.createElement('div');
			label.className = 'label-player';
			label.id = 'label-player-' + i;
			label.innerHTML = name;
			label.style.setProperty('left', (players[i].x * this.zoomScale - 75) + 'px' );
			label.style.setProperty('top', (players[i].y * this.zoomScale - 31) + 'px' );
			this.container.appendChild(label);
			// Add highlight function
			(function(i) { marker.onmouseover = function() { self.highlightPlayer(i); }; })(i);
			marker.onmouseout = function() { self.highlightNone(); };
		}
		
	};
	
	this.removePlayerMarkers = function() {
	
		var markers = [].slice.call(document.getElementsByClassName('marker-player'));
		var labels  = [].slice.call(document.getElementsByClassName('label-player'));
		for( var i = 0; i < markers.length; i++ ) {
			this.container.removeChild(markers[i]);
			this.container.removeChild(labels[i]);
		} 

	};
	
	this.highlightPlayer = function( player_id ) {
		var markers = document.getElementsByClassName('marker-player');
		for( var j = 0; j < markers.length; j++ ) {
			if( markers[j].id !== 'marker-player-' + player_id ) {
				markers[j].style.opacity = 0.2;
			}
		}
		var labels = document.getElementsByClassName('label-player');
		for( var j = 0; j < labels.length; j++ ) {
			if( labels[j].id !== 'label-player-' + player_id ) {
				labels[j].style.opacity = 0.2;
			}
		}
	};
	
	this.highlightNone = function() {
		var markers = document.getElementsByClassName('marker-player');
		for( var j = 0; j < markers.length; j++ ) markers[j].style.opacity = 1;
		var labels = document.getElementsByClassName('label-player');
		for( var j = 0; j < labels.length; j++ ) labels[j].style.opacity = 1;
	
	};
	
	this.highlightClaim = function( guild_id ) {

		this.drawGuildStandings(guild_id);
		this.standLayer.style.display = 'block';
		
	}
	
	this.unhighlightClaims = function() {

		this.standLayer.style.display = 'none';
		
	}
	
	this.drawGuildStandings = function( guild_id ) {
		
		if( this.zoomScale > 4 ) return false;
		if( ! Config.showStandings ) return false;
		
		canvas = this.standLayer;
		source = this.getGuildInfoById(guild_id);
		
		// Adjust canvas size
		canvas.width  = 1533 * this.zoomScale;
		canvas.height = 1533 * this.zoomScale;
		
		// Initialize 
		var ctx = canvas.getContext("2d");
		ctx.clearRect( 0, 0, 1533*this.zoomScale ,1533*this.zoomScale );
		
		// Draw line to each other claim
		for( var i = 0; i < claims.length; i++ ) {
			
			ctx.beginPath();
			
			// Skip self
			if( claims[i].id === guild_id ) continue;
			
			// Set default (war) standing color
			ctx.strokeStyle = 'Red';
			// Find non-default standings
			for( var j = 0; j < standings.length; j++ ) {
				// Skip non-relevant
				if( parseInt(standings[j].GuildID1) !== guild_id || parseInt(standings[j].GuildID2) !== claims[i].id ) continue;
				// Adjust color on match
				switch( parseInt(standings[j].StandingTypeID) ) {
					case 1:	ctx.strokeStyle = 'Red';			break;
					case 2: ctx.strokeStyle = 'Orange';			break;
					case 3: ctx.strokeStyle = 'White';			break;
					case 4: ctx.strokeStyle = 'YellowGreen';	break;
					case 5: ctx.strokeStyle = 'Lime';			break;
				}
			}
			
			ctx.moveTo( source.center_x * this.zoomScale, source.center_y * this.zoomScale );
			ctx.lineTo( claims[i].center_x * this.zoomScale, claims[i].center_y * this.zoomScale );
			ctx.stroke();
			
			ctx.closePath();

		}

	}
	
	// Toggle a canvas layer
	this.toggleLayer = function( layer ) {
		
		switch( layer ) {
			case 'structures':
				$('#ctrlbtn-structures').toggleClass('control-button-inactive');
				var canvas = this.buildLayer;
				var keyword = 'get_structures';
				var data = this.structures;
			break;
			case 'roads':
				$('#ctrlbtn-roads').toggleClass('control-button-inactive');
				var canvas = this.pavedLayer;
				var keyword = 'get_paved';
				var data = this.pavedTiles;
			break;
			case 'personal_claims':
				$('#ctrlbtn-personal-claims').toggleClass('control-button-inactive');
				$('.personal-claim').toggle();
				return true;
			break;
			case 'admin_lands':
				$('#ctrlbtn-admin-lands').toggleClass('control-button-inactive');
				$('.admin-land').toggle();
				return true;
			break;
			case 'regions':
				if( this.regions.length < 9 ) {
					$.getJSON( 'index.php?ajax=get_regions&livemap_id=' + livemap_id, function(data) {
						self.regions = data;
						$.each( data, function(index, terrainblock) {
							var TerID   = parseInt(terrainblock.ID);
							var TerSize = Math.round( $(self.container).width() / 3 );
							var posB = Math.floor((TerID-442) / 3) * TerSize;
							var posL = ((TerID-442) % 3) * TerSize;
							var css  = { 
								bottom: posB + 1,
								left: posL + 1, 
								width: (TerSize - 2) + 'px',
								height: (TerSize - 2) + 'px'
							}
							var rname = Text['reg'+terrainblock.RegionID];
							$("<div class='region-box region-" + terrainblock.RegionID + "'><p>" + rname + "</p></div>").css(css).appendTo( $(self.container) );
						} );
					} );
				} else {
					$(".region-box").toggle();
				}
				return true;
			case 'claim_planner':
				// Planner mode variables
				var outpost_opa = 0.1;
				var outpost_col = 'Orange';
				var claim_opa = 0.35;
				var claim_rad = 75;
				this.mouseR = 75;
				if( this.plannerBool ) {
					// Outposts Mode
					outpost_opa = 0.35;
					outpost_col = 'Red';
					claim_opa = 0.35;
					claim_rad = 150;
					this.mouseR = 150;
				}
				// Draw max claim shadows
				$(this.planLayer).toggle();
				$(this.mouseLayer).toggle();
				var ctx = this.planLayer.getContext('2d');
				ctx.clearRect(0,0,1533*this.zoomScale,1533*this.zoomScale);
				ctx.fillStyle = 'Red';
				$.each( claims, function(i, claim) {
					ctx.globalAlpha = 0.35;
					ctx.beginPath();
					ctx.arc(claim.center_x * self.zoomScale, claim.center_y * self.zoomScale, claim_rad * self.zoomScale, 0, 2*Math.PI);
					ctx.fill();
					ctx.closePath();
					if( ! self.plannerBool ) {
						ctx.globalAlpha = claim_opa;
						ctx.beginPath();
						ctx.arc(claim.center_x * self.zoomScale, claim.center_y * self.zoomScale, 50 * self.zoomScale, 0, 2*Math.PI);
						ctx.fill();
						ctx.closePath();
					}
				} );
				ctx.save();
				ctx.strokeStyle = outpost_col;
				ctx.fillStyle = outpost_col;
				ctx.setLineDash([5, 3]);
				$.each( outposts, function(i, outpost) {
					ctx.beginPath();
					ctx.arc(outpost.center_x * self.zoomScale, outpost.center_y * self.zoomScale, 150 * self.zoomScale, 0, 2*Math.PI);
					ctx.globalAlpha = 0.7;
					if( ! self.plannerBool ) ctx.stroke();
					ctx.globalAlpha = outpost_opa;
					ctx.fill();
					ctx.closePath();
				} );
				ctx.restore();
				// Attach mousemove handler
				if( this.mouseLayer.style.display !== 'none' ) {
					$(this.container).on( "mousemove", function(e) {
						self.mouseX = e.pageX - $(this).offset().left;
						self.mouseY = e.pageY - $(this).offset().top;
						self.drawMouseLayer();
					} );
					$(this.container).on( "click", function(e) {
						self.plannerBool = ! self.plannerBool;
						self.toggleLayer('claim_planner');
						self.toggleLayer('claim_planner');
						self.drawMouseLayer();
					} );
				} else {
					$(this.container).off( "mousemove" );
					$(this.container).off( "click" );
				}
				return true;
			break;
		}

		// Hide or show
		canvas.style.display = canvas.style.display === 'none' ? 'block' : 'none';
		
		// Stop here if hidden
		if( canvas.style.display === 'none' ) return true;

		// Pull data from server if local array is empty
		if( Object.keys(data).length < 1 ) {
			
			// Show the loading animation meanwhile
			this.showLoader();
	
			var post = new FormData();
			post.append('ajax', keyword);
			
			ajax_send( post, function(response) {
				
				try {
					// Transfer to local array reference and draw
					data = JSON.parse( response );
				} catch(e) {
					// On Error
					var mem_error = /exhausted/;
					var time_error = /execution/;
					if( mem_error.test(response) ) alert("Couldn't load data from server. Memory limit exceeded.");
					if( time_error.test(response) ) alert("Couldn't load data from server. Maximum script execution time exceeded.");
					self.hideLoader();
					return false;
				}
				
				// Structure layer data transfer
				if( layer === 'structures' ) {
					self.structures = data;
					self.drawLayer( canvas, data, 'Aqua', true );
				// Roads layer data transfer
				} else if( layer === 'roads' ) {
					self.pavedTiles = data;
					self.drawRoadLayer();
				}

				// Hide the loading animation when done
				self.hideLoader();
				
			} );
			
		// ... or just redraw the image if data is already in place
		} else {
			
			if( layer === 'structures' ) this.drawLayer( canvas, data, 'Aqua', true );
			else if( layer === 'roads' ) this.drawRoadLayer();
			
		}

	};
	
	this.drawMouseLayer = function() {
		
		// Check for collision
		var collision = false;
		$.each( outposts, function(i, outpost) {
			var distance = Math.sqrt( Math.pow(self.mouseX - outpost.center_x * self.zoomScale, 2) + Math.pow(self.mouseY - outpost.center_y * self.zoomScale, 2) );
			if( distance <= 150 * self.zoomScale ) collision = true;
		} );
		$.each( claims, function(i, claim) {
			var distance = Math.sqrt( Math.pow(self.mouseX - claim.center_x * self.zoomScale, 2) + Math.pow(self.mouseY - claim.center_y * self.zoomScale, 2) );
			if( self.plannerBool && distance <= 150 * self.zoomScale ) collision = true;
			if( ! self.plannerBool && distance <= claim.radius * self.zoomScale ) collision = true;
		} );

		var ctx = self.mouseLayer.getContext('2d');
		ctx.textBaseline = 'middle';
		ctx.textAlign = 'center';
		ctx.fillStyle = collision ? 'Orange' : 'Lime';
		ctx.clearRect(0,0,1533*self.zoomScale,1533*self.zoomScale);
		ctx.beginPath();
		ctx.globalAlpha = collision ? 0.5 : 0.3;
		ctx.arc(self.mouseX, self.mouseY, this.mouseR * self.zoomScale, 0, 2*Math.PI);
		ctx.fill();
		ctx.closePath();
		if( ! self.plannerBool ) {
			ctx.beginPath();
			ctx.globalAlpha = 0.3;
			ctx.arc(self.mouseX, self.mouseY, 50 * self.zoomScale, 0, 2*Math.PI);
			ctx.fill();
			ctx.closePath();
		}
		// Label
		ctx.globalAlpha = 1;
		ctx.fillStyle = 'Black';
		var circleType = this.plannerBool ? Text.otpst : "Monument";
		ctx.font = 'bold 12px sans-serif';
		ctx.fillText( circleType, self.mouseX, self.mouseY - 25 );
		ctx.font = '12px sans-serif';
		ctx.fillText( "click to toggle", self.mouseX, self.mouseY - 10 );
		if( collision ) {
			ctx.fillStyle = 'Maroon';
			ctx.font = 'bold 12px sans-serif';
			ctx.fillText( "BLOCKED", self.mouseX, self.mouseY + 30 );
		}
		
	}

	this.drawRoadLayer = function() {
		this.drawLayer( this.pavedLayer, this.pavedTiles.stone, 'Gray', true );
		this.drawLayer( this.pavedLayer, this.pavedTiles.slate, 'LightSlateGray', false );
		this.drawLayer( this.pavedLayer, this.pavedTiles.marble, 'Tan', false );
	};
	
	this.drawLayer = function( canvas, data, color, wipe ) {
		
		var ctx = canvas.getContext("2d");

		if( wipe ) {
			canvas.width  = 1533 * this.zoomScale;
			canvas.height = 1533 * this.zoomScale;
			ctx.clearRect( 0, 0, 1533*this.zoomScale, 1533*this.zoomScale );
		}
		
		ctx.fillStyle = color;
		
		for( var i = 0; i < data.length; i++ ) ctx.fillRect( data[i][0] * this.zoomScale, data[i][1] * this.zoomScale, Math.round(1*this.zoomScale), Math.round(1*this.zoomScale) );
 
	};
	
	this.toggleGrid = function() {
		
		// Adjust canvas size
		this.gridLayer.width   = 1533 * this.zoomScale;
		this.gridLayer.height  = 1533 * this.zoomScale;

		// Hide or show the element
		this.gridLayer.style.display = this.gridLayer.style.display === 'none' ? 'block' : 'none';
		
		// Stop here if hidden
		if( this.gridLayer.style.display === 'none' ) return true;
		
		// Calculate dimensions
		var size = 1533 * this.zoomScale;
		var divi = 25;
		var step = size / divi;
		var letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';

		// Init canvas
		var ctx = this.gridLayer.getContext("2d");
		ctx.clearRect( 0, 0, size, size );
		ctx.fillStyle = 'White';
		ctx.textAlign = 'center';
		ctx.textBaseline = 'middle';
		ctx.font = '12px serif';
		
		// Draw horizontal raster
		var y = 1;
		for( var ypos = step/2; ypos < size; ypos += step ) {
			ctx.fillRect( parseInt(step/2), parseInt(ypos), parseInt(size-step), this.zoomScale );
			if( parseInt(ypos+step/2) < size-5 ) ctx.fillText( y++, parseInt(step/4), parseInt(ypos+step/2) );
		}
		
		// Draw vertical raster
		var x = 0;
		for( var xpos = step/2; xpos < size; xpos += step ) {
			ctx.fillRect( parseInt(xpos), parseInt(step/2), this.zoomScale, parseInt(size-step) );
			if( parseInt(xpos+step/2) < size-5 ) ctx.fillText( letters.charAt(x++), parseInt(xpos+step/2), parseInt(step/4) );
		}

	};
	
	// Toggle map background image
	this.toggleImage = function() {
		// Flip images
		this.defaultImage = !this.defaultImage;
		this.container.style.backgroundImage = this.defaultImage ? "url('" + Config.defaultMap + "')" : "url('" + Config.alternativeMap + "')";
		// Save config in cookie
		if( this.defaultImage ) setCookie('customMapStatus', '1', 30);
		else setCookie('customMapStatus', '0', 30);
	};
	
	// Zoom in or out
	this.zoom = function( zoomin ) {
		
		// Calculate target scale
		var newScale = zoomin ? this.zoomScale * 2 : this.zoomScale / 2;
		
		// Limits
		if( newScale >= 8.1 || newScale <= 0.4 ) return false;
		
		// Get current viewport center
		var clientw = window.innerWidth;
		var clienth = window.innerHeight;
		var mapsize = 1533 * this.zoomScale;
		if( clientw >= mapsize ) var centerX = Math.round( mapsize / 2 );
		else					 var centerX = Math.round( parseInt(document.body.scrollLeft) + parseInt(document.documentElement.scrollLeft) + clientw/2 );
		if( clienth >= mapsize ) var centerY = Math.round( mapsize / 2 );
		else					 var centerY = Math.round( parseInt(document.body.scrollTop) + parseInt(document.documentElement.scrollTop) + clienth/2 );
		
		// Apply new scale
		this.zoomScale = newScale;
		var mult = zoomin ? 2 : 0.5;
		
		// If zooming out, do the scrolling first. 
		if( ! zoomin ) window.scrollBy( -(centerX/2), -(centerY/2) );
		
		// Zoom: Livemap container ...
		this.container.style.width  = Math.round( 1533 * this.zoomScale ) + 'px';
		this.container.style.height = Math.round( 1533 * this.zoomScale ) + 'px';
		this.container.style.backgroundSize = '100%';
		
		// Zoom layers
		this.planLayer.width   = 1533 * this.zoomScale;
		this.planLayer.height  = 1533 * this.zoomScale;
		this.mouseLayer.width  = 1533 * this.zoomScale;
		this.mouseLayer.height = 1533 * this.zoomScale;
		
		// Zoom: Guild claims ...
		var claims = document.getElementsByClassName('guild-claim');
		for( var i = 0; i < claims.length; i++ ) {
			var x = claims[i].style.left.slice(0, -2);
			var y = claims[i].style.top.slice(0, -2);
			var d = claims[i].style.width.slice(0, -2);
			claims[i].style.left   = parseInt(x*mult) + 'px';
			claims[i].style.top    = parseInt(y*mult) + 'px';
			claims[i].style.width  = parseInt(d*mult) + 'px';
			claims[i].style.height = parseInt(d*mult) + 'px';
			claims[i].style.borderRadius = parseInt(d*mult*2) + 'px';
		}
		
		// Zoom: Personal claims and admin lands ...
		$('.personal-claim, .admin-land').each( function() {
			$(this).width( $(this).width() * mult );
			$(this).height( $(this).height() * mult );
			$(this).css( {top: ($(this).position().top * mult) + 'px', left: ($(this).position().left * mult) + 'px'} );
		} );

		// Zoom: Claim top labels ...	
		var tl = [].slice.call(document.getElementsByClassName('guild-name-top'));
		for( var i = 0; i < tl.length; i++ ) {
			var x = tl[i].style.left.slice(0, -2);
			var y = tl[i].style.top.slice(0, -2);
			var w = tl[i].style.width.slice(0, -2);
			if( zoomin ) {
				tl[i].style.left = parseInt(x*2+parseInt(w)/2) + 'px';
				tl[i].style.top  = parseInt(y*2+16) + 'px';
			} else {
				tl[i].style.left = parseInt(x/2-parseInt(w)/4) + 'px';
				tl[i].style.top  = parseInt(y/2-8) + 'px';
			}
		}
		
		// Zoom: Claim center labels ...
		var cl = [].slice.call(document.getElementsByClassName('guild-name-center'));
			for( var i = 0; i < cl.length; i++ ) {
			var x = cl[i].style.left.slice(0, -2);
			var y = cl[i].style.top.slice(0, -2);
			var w = cl[i].style.width.slice(0, -2);
			var h = cl[i].style.height.slice(0, -2);
			if( zoomin ) {
				cl[i].style.left = parseInt(x*2+parseInt(w)/2) + 'px';
				cl[i].style.top  = parseInt(y*2+parseInt(h)/2) + 'px';
			} else {
				cl[i].style.left = parseInt(x/2-parseInt(w)/4) + 'px';
				cl[i].style.top  = parseInt(y/2-parseInt(h)/4) + 'px';
			}
		}
		
		// Zoom markers
		$(".map-marker-16").each( function(index, element) {
			var currentpos = $(element).position();
			$(element).css( {
				top: (currentpos.top * mult) + 'px',
				left: (currentpos.left * mult) + 'px'
			} );
		} );
		
		// Region boxes
		$(".region-box").each( function(index, element) {
			var posB = parseInt( $(element).css('bottom'), 10);
			var posL = parseInt( $(element).css('left'), 10);
			$(element).css( {
				bottom: posB * mult,
				left: posL * mult,
				width: ($(element).width() * mult) + 'px',
				height: ($(element).height() * mult) + 'px'
			} );
		} );
		
		// Players ...
		if( this.showPlayers ) { 
			this.removePlayerMarkers();
			this.drawPlayerMarkers();
		}
		
		// Road and structure layers ...
		if( this.buildLayer.style.display === 'block' ) this.drawLayer( this.buildLayer, this.structures, 'Cyan', true );
		if( this.pavedLayer.style.display === 'block' ) this.drawRoadLayer();
		if( this.gridLayer.style.display === 'block' ) {
			this.toggleGrid();
			this.toggleGrid();
		}
		
		if( this.planLayer.style.display !== 'none' ) {
			this.toggleLayer('claim_planner');
			this.toggleLayer('claim_planner');
		}
		
		// Scroll window to new position after zooming in
		if( zoomin ) window.scrollBy( centerX, centerY );

	};
	
	// Enable disable zooming with mouse wheel
	this.toggleWheel = function() {
		this.wheelZoom = ! this.wheelZoom;
		document.getElementById('mousewheel-status-icon').src = this.wheelZoom ? 'img/ni-mousewheel-active.png' : 'img/ni-mousewheel-disabled.png';
		setCookie('wheelZoom', '0', 30);
	}
	
	// Detect cookie settings
	if( getCookie('customMapStatus') === '0' && Config.altMapEnabled ) this.toggleImage();
	if( getCookie('gridStatus')      === '1' ) this.toggleGrid();
	if( getCookie('wheelZoom')		 === '0' ) this.toggleWheel();
	
	// Attach events for drag-scrolling
	window.addEventListener('mousemove', function(e) { 
		if(this.curDown) {
			e = window.event || e; 
			document.body.style.cursor = "move";
			// Chrome, Edge, Opera, Safari
			document.body.scrollLeft += (this.curXPos - e.pageX);
			document.body.scrollTop  += (this.curYPos - e.pageY);
			// Internet Explorer, Firefox
			document.documentElement.scrollLeft += (this.curXPos - e.pageX);
			document.documentElement.scrollTop  += (this.curYPos - e.pageY);
		}
	});
	window.addEventListener('mousedown', function(e) {
		e = window.event || e; 
		this.curYPos = e.pageY;
		this.curXPos = e.pageX; 
		this.curDown = true;
	});
	window.addEventListener('mouseup', function(e) {
		e = window.event || e; 
		this.curDown = false;
		document.body.style.cursor = 'auto';
	});
	window.onwheel = function(e){
		if( ! self.wheelZoom ) return true;
		e = window.event || e;
		if( e.deltaY < 0 ) self.zoom(true);
		if( e.deltaY > 1 ) self.zoom(false);
		return false;
	}

}